<?php

namespace Drupal\Tests\myacademicid_user_roles\Functional;

use Drupal\user\Entity\User;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\Role;

/**
 * Tests the Role mapping form.
 *
 * @group myacademicid_user_fields
 */
class RoleMappingFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'oauth2_server',
    'myacademicid_user_fields',
    'myacademicid_user_roles',
  ];

  /**
   * Admin user account.
   */
  protected User $admin;

  /**
   * Authenticated user account.
   */
  protected User $user;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set Server mode by default for these tests.
    \Drupal::configFactory()
      ->getEditable('myacademicid_user_fields.settings')
      ->set('mode', 'server')
      ->save();

    // New admin user, new admin role which does not interfere with the test.
    $permissions = ['administer myacademicid user fields'];
    $this->admin = $this->drupalCreateUser($permissions, 'boss', TRUE, []);

    // New user, new role which interferes with the test and must be removed.
    $this->user = $this->drupalCreateUser([]);
    foreach ($this->user->getRoles(TRUE) as $rid) {
      $this->user->removeRole($rid);
    }

    // Clean up autogenerated roles.
    $roles = Role::loadMultiple();
    foreach ($roles as $rid => $role) {
      $protected = in_array($rid, ['anonymous', 'authenticated']);
      if (!$protected && !$role->isAdmin()) {
        $role->delete();
      }
    }

    // Define a named role for testing.
    Role::create([
      'id' => 'content_editor',
      'label' => 'Content editor',
    ])->save();

  }

  /**
   * Tests access to the Role mapping form by a non-privileged user.
   */
  public function testRoleMappingFormWithoutPermission() {
    $this->drupalLogin($this->user);

    $this->drupalGet('admin/config/services/myacademicid/role-mapping');
    $this->assertSession()
      ->statusCodeEquals(403);
  }

  /**
   * Tests the Role mapping form as a privileged user.
   */
  public function testRoleMappingForm() {
    $this->drupalLogin($this->admin);

    // Test access to the Role mapping form.
    $this->drupalGet('admin/config/services/myacademicid/role-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);
    // Test the default configuration.
    $default_config = $this->config('myacademicid_user_roles.role_to_affiliation')
      ->get('role_mapping');
    $this->assertEmpty($default_config);

    // Test form field exists for the named role.
    $this->assertSession()
      ->fieldExists('edit-role-mapping-content-editor');
    $this->assertSession()
      ->pageTextContainsOnce('Content editor');

    // Test form field options include the default affiliation types.
    $this->assertSession()
      ->optionExists('edit-role-mapping-content-editor', 'staff');
    $this->assertSession()
      ->pageTextMatchesCount(3, '/Staff/');

    // Test form field does not exist for anonymous.
    $this->assertSession()
      ->fieldNotExists('edit-role-mapping-anonymous');

    // Test form submission.
    $submission_data = ['edit-role-mapping-content-editor' => 'staff'];
    $this->submitForm($submission_data, 'Save configuration');
    $this->assertSession()
      ->pageTextContains('The configuration options have been saved.');

    // Test the configuration has been updated.
    $new_config = $this->config('myacademicid_user_roles.role_to_affiliation')
      ->get('role_mapping');
    $this->assertEquals($new_config, ['content_editor' => 'staff']);

    // Test the form loads normally.
    $this->drupalGet('admin/config/services/myacademicid/role-mapping');
    $element = $this->assertSession()
      ->optionExists('edit-role-mapping-content-editor', 'staff');
    $this->assertEquals('selected', $element->getAttribute('selected'));

  }

  /**
   * Tests the Role mapping form with additional affiliation types.
   */
  public function testRoleMappingFormAdditionalTypes() {
    $this->drupalLogin($this->admin);

    // Set additional affiliation types in upstream config.
    $additional = [
      'ewp-admin',
      'honor|Honoris Causa',
      'faculty|Academic staff',
    ];
    \Drupal::configFactory()
      ->getEditable('myacademicid_user_fields.types')
      ->set('additional', $additional)
      ->save();

    // Test access to the Role mapping form.
    $this->drupalGet('admin/config/services/myacademicid/role-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);

    // Test form field options exist for the additional affiliation claims.
    $this->assertSession()
      ->optionExists('edit-role-mapping-content-editor', 'ewp-admin');
    $this->assertSession()
      ->optionExists('edit-role-mapping-content-editor', 'honor');

    // Test form field handles affiliation label override.
    $element = $this->assertSession()
      ->optionExists('edit-role-mapping-content-editor', 'faculty');
    $this->assertEquals('Academic staff', $element->getText());

    // Test form submission.
    $submission_data = ['edit-role-mapping-content-editor' => 'faculty'];
    $this->submitForm($submission_data, 'Save configuration');
    $this->assertSession()
      ->pageTextContains('The configuration options have been saved.');

    // Test the configuration has been updated.
    $new_config = $this->config('myacademicid_user_roles.role_to_affiliation')
      ->get('role_mapping');
    $this->assertEquals($new_config, ['content_editor' => 'faculty']);

    // Test the form loads normally.
    $this->drupalGet('admin/config/services/myacademicid/role-mapping');
    $element = $this->assertSession()
      ->optionExists('edit-role-mapping-content-editor', 'faculty');
    $this->assertEquals('selected', $element->getAttribute('selected'));

  }

  /**
   * Tests the Role mapping form in Client mode.
   */
  public function testRoleMappingFormClientMode() {
    $this->drupalLogin($this->admin);

    // Set Client mode in upstream config.
    \Drupal::configFactory()
      ->getEditable('myacademicid_user_fields.settings')
      ->set('mode', 'client')
      ->save();

    // Test access to the Role mapping form.
    $this->drupalGet('admin/config/services/myacademicid/role-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);

    // Test the Client mode enabled triggers a warning on the page.
    $this->assertSession()
      ->statusMessageExists('warning');
    // Check for the plain text first.
    $this->assertSession()
      ->pageTextContains('These settings have no effect when operating in');
    // Check for the content of the anchor tag separately.
    $this->assertSession()
      ->pageTextContains('Client mode');
  }

}
