<?php

namespace Drupal\Tests\myacademicid_user_roles\Functional;

use Drupal\user\Entity\User;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\Role;

/**
 * Tests the Affiliation mapping form.
 *
 * @group myacademicid_user_fields
 */
class AffiliationMappingFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'oauth2_server',
    'myacademicid_user_fields',
    'myacademicid_user_roles',
  ];

  /**
   * Admin user account.
   */
  protected User $admin;

  /**
   * Authenticated user account.
   */
  protected User $user;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // New admin user, new admin role which does not interfere with the test.
    $permissions = ['administer myacademicid user fields'];
    $this->admin = $this->drupalCreateUser($permissions, 'boss', TRUE, []);

    // New user, new role which interferes with the test and must be removed.
    $this->user = $this->drupalCreateUser([]);
    foreach ($this->user->getRoles(TRUE) as $rid) {
      $this->user->removeRole($rid);
    }

    // Clean up autogenerated roles.
    $roles = Role::loadMultiple();
    foreach ($roles as $rid => $role) {
      $protected = in_array($rid, ['anonymous', 'authenticated']);
      if (!$protected && !$role->isAdmin()) {
        $role->delete();
      }
    }

    // Define a named role for testing.
    Role::create([
      'id' => 'content_editor',
      'label' => 'Content editor',
    ])->save();
  }

  /**
   * Tests access to the Affiliation mapping form by a non-privileged user.
   */
  public function testAffiliationMappingFormWithoutPermission() {
    $this->drupalLogin($this->user);

    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $this->assertSession()
      ->statusCodeEquals(403);
  }

  /**
   * Tests the Affiliation mapping form as a privileged user.
   */
  public function testAffiliationMappingForm() {
    $this->drupalLogin($this->admin);

    // Test access to the Affiliation mapping form.
    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);

    // Test the default configuration.
    $default_config = $this->config('myacademicid_user_roles.affiliation_to_role')
      ->get('affiliation_mapping');
    $this->assertEmpty($default_config);

    // Test form field exists for a default affiliation claim.
    $this->assertSession()
      ->fieldExists('edit-affiliation-mapping-staff');
    $this->assertSession()
      ->pageTextContainsOnce('Staff');

    // Test form field options include the single eligible role.
    $this->assertSession()
      ->optionExists('edit-affiliation-mapping-staff', 'content_editor');

    // Test form field options do not include protected roles.
    $this->assertSession()
      ->optionNotExists('edit-affiliation-mapping-staff', 'anonymous');
    $this->assertSession()
      ->optionNotExists('edit-affiliation-mapping-staff', 'authenticated');

    // Test form field options do not include admin role.
    $admin_rid = array_keys($this->admin->getRoles(TRUE))[0];
    $this->assertSession()
      ->optionNotExists('edit-affiliation-mapping-staff', $admin_rid);

    // Test form submission.
    $submission_data = ['edit-affiliation-mapping-staff' => 'content_editor'];
    $this->submitForm($submission_data, 'Save configuration');
    $this->assertSession()
      ->pageTextContains('The configuration options have been saved.');

    // Test the configuration has been updated.
    $new_config = $this->config('myacademicid_user_roles.affiliation_to_role')
      ->get('affiliation_mapping');
    $this->assertEquals($new_config, ['staff' => 'content_editor']);

    // Test the form loads normally.
    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $element = $this->assertSession()
      ->optionExists('edit-affiliation-mapping-staff', 'content_editor');
    $this->assertEquals('selected', $element->getAttribute('selected'));

  }

  /**
   * Tests the Affiliation mapping form with additional affiliation types.
   */
  public function testAffiliationMappingFormAdditionalTypes() {
    $this->drupalLogin($this->admin);

    // Set additional affiliation types in upstream config.
    $additional = [
      'ewp-admin',
      'honor|Honoris Causa',
      'faculty|Academic staff',
    ];
    \Drupal::configFactory()
      ->getEditable('myacademicid_user_fields.types')
      ->set('additional', $additional)
      ->save();

    // Test access to the settings form.
    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);

    // Test form field exists for the additional affiliation claims.
    $this->assertSession()
      ->fieldExists('edit-affiliation-mapping-ewp-admin');
    $this->assertSession()
      ->pageTextContainsOnce('ewp-admin');
    $this->assertSession()
      ->fieldExists('edit-affiliation-mapping-honor');
    $this->assertSession()
      ->pageTextContainsOnce('Causa');

    // Test form field handles affiliation label override.
    $this->assertSession()
      ->fieldExists('edit-affiliation-mapping-faculty');
    $this->assertSession()
      ->pageTextContainsOnce('Academic staff');

    // Test form field options include the single eligible role.
    $this->assertSession()
      ->optionExists('edit-affiliation-mapping-honor', 'content_editor');

    // Test form field options do not include protected roles.
    $this->assertSession()
      ->optionNotExists('edit-affiliation-mapping-honor', 'anonymous');
    $this->assertSession()
      ->optionNotExists('edit-affiliation-mapping-honor', 'authenticated');

    // Test form field options do not include admin role.
    $admin_rid = array_keys($this->admin->getRoles(TRUE))[0];
    $this->assertSession()
      ->optionNotExists('edit-affiliation-mapping-honor', $admin_rid);

    // Test form submission.
    $submission_data = ['edit-affiliation-mapping-honor' => 'content_editor'];
    $this->submitForm($submission_data, 'Save configuration');
    $this->assertSession()
      ->pageTextContains('The configuration options have been saved.');

    // Test the configuration has been updated.
    $new_config = $this->config('myacademicid_user_roles.affiliation_to_role')
      ->get('affiliation_mapping');
    $this->assertEquals($new_config, ['honor' => 'content_editor']);

    // Test the form loads normally.
    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $element = $this->assertSession()
      ->optionExists('edit-affiliation-mapping-honor', 'content_editor');
    $this->assertEquals('selected', $element->getAttribute('selected'));

  }

  /**
   * Tests the Affiliation mapping form in Server mode.
   */
  public function testAffiliationMappingFormServerMode() {
    $this->drupalLogin($this->admin);

    // Set Server mode in upstream config.
    \Drupal::configFactory()
      ->getEditable('myacademicid_user_fields.settings')
      ->set('mode', 'server')
      ->save();

    // Test access to the Affiliation mapping form.
    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);

    // Test the Server mode enabled triggers a warning on the page.
    $this->assertSession()
      ->statusMessageExists('warning');
    // Check for the plain text first.
    $this->assertSession()
      ->pageTextContains('These settings have no effect when operating in');
    // Check for the content of the anchor tag separately.
    $this->assertSession()
      ->pageTextContains('Server mode');
  }

  /**
   * Tests the Affiliation mapping form when there are no eligible roles.
   */
  public function testAffiliationMappingFormNoRoles() {
    $this->drupalLogin($this->admin);

    // Undo setup: delete the only eligible role.
    Role::load('content_editor')->delete();

    // Test access to the Affiliation mapping form.
    $this->drupalGet('admin/config/services/myacademicid/affiliation-mapping');
    $this->assertSession()
      ->statusCodeEquals(200);

    // Test the lack of eligible roles triggers a warning on the page.
    $this->assertSession()
      ->statusMessageExists('warning');
    $this->assertSession()
      ->pageTextContains('There are no user roles available for mapping.');

    // Test the lack of eligible roles results in an empty form.
    $this->assertSession()
      ->buttonNotExists('Save configuration');
  }

}
