<?php

namespace Drupal\eu_funding\Plugin\Block;

use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a block with the EU funding banner.
 */
#[Block(
  id: 'eu_funding_banner',
  admin_label: new TranslatableMarkup("EU funding banner"),
  category: new TranslatableMarkup("EU funding"),
)]
class FundingBanner extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->stringTranslation = $container->get('string_translation');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $module_path = $this->moduleHandler->getModule('eu_funding')->getPath();

    $banner_dir = '/' . $module_path . '/img';

    $banner_default = 'eu_flag_co_funded.png';
    $banner_high_contrast = 'eu_flag_co_funded_hc.png';

    if ($this->configuration['high_contrast'] == 1) {
      $banner = $banner_dir . '/' . $banner_high_contrast;
    }
    else {
      $banner = $banner_dir . '/' . $banner_default;
    }

    $banner_alt = $this->t('Co-funded by the Erasmus+ Programme of the European Union');

    return [
      '#theme' => 'eu_funding_banner',
      '#banner' => $banner,
      '#alttext' => $banner_alt,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $form['high_contrast'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('High contrast'),
      '#description' => $this->t('Check this option if the banner will be placed over a dark background.'),
      '#default_value' => $this->configuration['high_contrast'] ?? 0,
      '#return_value' => 1,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $values = $form_state->getValues();
    $this->configuration['high_contrast'] = $values['high_contrast'];
  }

}
