<?php

namespace Drupal\erasmus_subject_area_code\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\OptGroup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\erasmus_subject_area_code\ErasmusSubjectAreaCodeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'erasmus_subject_area_code_default' formatter.
 */
#[FieldFormatter(
  id: 'erasmus_subject_area_code_default',
  label: new TranslatableMarkup('Default'),
  field_types: [
    'erasmus_subject_area_code',
  ],
)]
class ErasmusSubjectAreaCodeDefaultFormatter extends FormatterBase {

  const SEPARATOR = ' ';

  /**
   * Erasmus Subject Area code manager.
   *
   * @var \Drupal\erasmus_subject_area_code\ErasmusSubjectAreaCodeManagerInterface
   */
  protected $codeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    ErasmusSubjectAreaCodeManagerInterface $code_manager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->codeManager = $code_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('erasmus_subject_area_code.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $setting = ['prefix' => TRUE];
    return $setting + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements['prefix'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Prefix the subject area label with its code'),
      '#default_value' => $this->getSetting('prefix'),
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    return [
      $this->t('Prefix the subject area label with its code: @bool', [
        '@bool' => $this->getSetting('prefix')
          ? $this->t('Yes')
          : $this->t('No'),
      ]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $prefix = $this->getSetting('prefix');
    $options = $this->codeManager->getSelectOptions($prefix);
    $mapping = OptGroup::flattenOptions($options);
    $elements = [];
    foreach ($items as $delta => $item) {
      $label = $mapping[$item->value] ?? $item->value;
      $elements[$delta] = ['#markup' => $label];
    }
    return $elements;
  }

}
